/* Filename: IndicationDetailsPage.java
 * Creator: Raquel Hervas
 * Format: Java 2 v1.6.0
 * Date created: 23/09/2009
 */
package nil.ucm.indications2.ui.helpers;

import nil.ucm.indications2.core.rep.IIndicationStructure;
import nil.ucm.indications2.core.rep.IModifier;
import nil.ucm.indications2.core.rep.INucleus;
import nil.ucm.indications2.core.rep.IndicationStructureRep;
import nil.ucm.indications2.core.rep.IndicationType;
import nil.ucm.indications2.core.rep.change.ModifyModifierTypeChange;
import nil.ucm.indications2.core.rep.change.ModifyNucleusTypeChange;
import nil.ucm.indications2.core.rep.change.RemoveModifierChange;
import nil.ucm.indications2.core.rep.change.RemoveNucleusChange;
import nil.ucm.indications2.core.rep.change.SetCopularPredicateChange;
import nil.ucm.indications2.ui.content.IndicationStructureContentProvider;
import nil.ucm.indications2.ui.content.IndicationStructureContentProvider.ITypedList;
import nil.ucm.indications2.ui.content.IndicationStructureContentProvider.ReferentReferencePair;
import nil.ucm.indications2.ui.label.IndicationStructureLabelProvider;

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.viewers.IPostSelectionProvider;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.actions.ActionFactory;
import org.eclipse.ui.actions.ActionGroup;
import org.eclipse.ui.texteditor.ITextEditorActionConstants;

import edu.mit.discourse.core.rep.referent.ReferentRep;
import edu.mit.discourse.core.rep.referent.model.IReferenceModel;
import edu.mit.discourse.ui.IDiscourseUIConstantIDs;
import edu.mit.discourse.ui.helpers.ReferentCreatorPage;
import edu.mit.story.core.desc.IDesc;
import edu.mit.story.core.meta.timing.ITiming;
import edu.mit.story.core.meta.timing.Timing;
import edu.mit.story.core.model.change.IModelListener;
import edu.mit.story.core.model.change.RemoveDescriptionsChange;
import edu.mit.story.core.model.change.StoryChangeEvent;
import edu.mit.story.ui.IStoryUIConstantIDs;
import edu.mit.story.ui.WorkbenchUtils;
import edu.mit.story.ui.actions.ActionUtils;
import edu.mit.story.ui.actions.SelectionUtils;
import edu.mit.story.ui.actions.generic.CollapseAllAction;
import edu.mit.story.ui.actions.generic.DeleteAction;
import edu.mit.story.ui.actions.generic.EditAction;
import edu.mit.story.ui.actions.generic.ExpandAllAction;
import edu.mit.story.ui.actions.groups.StoryEditorUndoRedoActionGroup;
import edu.mit.story.ui.editor.IStoryEditor;
import edu.mit.story.ui.helpers.HelperPage;
import edu.mit.story.ui.helpers.ICreatorPage;
import edu.mit.story.ui.helpers.IDetailsPage;
import edu.mit.story.ui.viewers.StoryElementComparer;
import edu.mit.story.ui.views.CreatorView;
import edu.mit.story.ui.views.CreatorView.CreatorPageBookWidget;

/** 
 *   Details page for the IndicationStructure representations. It will present all the information
 * related with all the Indication Structures in the story, with a clear differentiation between
 * the different types of indications or references.
 *
 * @author Raquel Hervas
 * @version 1.0, (Jan. 08, 2010)
 * @since nil.ucm.indications.ui 1.0.0
 */
public class IndicationDetailsPage extends HelperPage implements IDetailsPage, IAdaptable,	IModelListener {
	
	public static final String ID = "nil.ucm.indications2.ui.helpers.IndicationDetailsPage";
	
	// widgets
	protected TreeViewer treeViewer;
	
	// actions
	EditIndicationAction editAction; 
	ToggleCopularAction toggleAction;
	IAction deleteAction;
	IAction collapseAction;
	IAction expandAction;
	ChangeIndicationTypeAction changeDEAction, changeDIAction, changeOTAction, changeUNAction;
	ActionGroup undoRedoActions;

	/* 
	 * (non-Javadoc) @see edu.mit.story.ui.editor.IHasStoryEditor#setStoryEditor(edu.mit.story.ui.editor.IStoryEditor)
	 */
	public void setStoryEditor(IStoryEditor newEditor) {
		super.setStoryEditor(newEditor);
		newEditor.getStoryModel().addModelListener(this);
	}

	/* 
	 * (non-Javadoc) @see org.eclipse.ui.part.IPage#createControl(org.eclipse.swt.widgets.Composite)
	 */
	
	public void createControl(Composite parent) {
		
		// create tree viewer
        treeViewer = new TreeViewer(parent, SWT.SINGLE | SWT.H_SCROLL | SWT.V_SCROLL);
        setControl(treeViewer.getTree());
        treeViewer.setComparer(StoryElementComparer.getInstance());
        treeViewer.setAutoExpandLevel(2);
        treeViewer.setContentProvider(new IndicationStructureContentProvider(getStoryEditor()));
        treeViewer.setLabelProvider(new IndicationStructureLabelProvider(getStoryEditor()));
        treeViewer.setInput(getStoryEditor());

		// create actions
		editAction = new EditIndicationAction();
		treeViewer.addSelectionChangedListener(editAction);
		
		toggleAction = new ToggleCopularAction();
		treeViewer.addSelectionChangedListener(toggleAction);
		
		collapseAction = new CollapseAllAction(treeViewer, 2);
		expandAction = new ExpandAllAction(treeViewer);

		deleteAction = new DeleteElementAction();
		undoRedoActions = new StoryEditorUndoRedoActionGroup(getStoryEditor());

		changeDEAction = new ChangeIndicationTypeAction(IndicationType.DESCRIPTIVE);
		changeDIAction = new ChangeIndicationTypeAction(IndicationType.DISTINCTIVE);
		changeOTAction = new ChangeIndicationTypeAction(IndicationType.OTHER);
		changeUNAction = new ChangeIndicationTypeAction(IndicationType.UNKNOWN);
		
		//Submenu for the change of type
		final MenuManager subMenu = new MenuManager("&Change Type");
		subMenu.add(changeDEAction);
		subMenu.add(changeDIAction);
		subMenu.add(changeOTAction);
		subMenu.add(changeUNAction);
		
		// create popup menu manager
		final MenuManager manager = new MenuManager();
		ActionUtils.createPopupGroups(manager);
		undoRedoActions.fillContextMenu(manager);
		manager.appendToGroup(ITextEditorActionConstants.GROUP_EDIT, editAction);
		manager.appendToGroup(ITextEditorActionConstants.GROUP_EDIT, toggleAction);
		manager.appendToGroup(ITextEditorActionConstants.GROUP_EDIT, subMenu);
		manager.appendToGroup(ITextEditorActionConstants.GROUP_EDIT, deleteAction);
		
		Menu menu = manager.createContextMenu(getControl());
		treeViewer.getTree().setMenu(menu);
		getSite().registerContextMenu(ID + ".popupMenu", manager, treeViewer);

	}

	/* 
	 * (non-Javadoc) @see org.eclipse.ui.part.IPage#dispose()
	 */
	public void dispose() {
		super.dispose();
		getStoryEditor().getStoryModel().removeModelListener(this);
	}

	/* 
	 * (non-Javadoc) @see org.eclipse.ui.part.IPage#setActionBars(org.eclipse.ui.IActionBars)
	 */
	
	public void setActionBars(IActionBars actionBars) {
		actionBars.getToolBarManager().insertBefore("linking", collapseAction);
		actionBars.getToolBarManager().insertBefore(collapseAction.getId(), expandAction);
		
		actionBars.setGlobalActionHandler(ActionFactory.DELETE.getId(), deleteAction);
		actionBars.setGlobalActionHandler(ActionFactory.RENAME.getId(), editAction);
		undoRedoActions.fillActionBars(actionBars);
		undoRedoActions.updateActionBars();
	}

	/* 
	 * (non-Javadoc) @see org.eclipse.core.runtime.IAdaptable#getAdapter(java.lang.Class)
	 */
	@SuppressWarnings("unchecked")
	public Object getAdapter(Class adapter) {
	
		// return requests for selection providers
		if(adapter == ISelectionProvider.class) return treeViewer;
		if(adapter == IPostSelectionProvider.class) return treeViewer;
		// otherwise defer to platform's adapter manager
		return Platform.getAdapterManager().getAdapter(this, adapter);
	}

	/* 
	 * (non-Javadoc) @see edu.mit.story.core.model.change.IModelListener#modelChanged(edu.mit.story.core.model.change.StoryChangeEvent)
	 */
	
	public void modelChanged(StoryChangeEvent e) {
		if(treeViewer == null) 
			return;
		if(e.affects(IndicationStructureRep.getInstance()) || e.affects(ReferentRep.getInstance())){
			treeViewer.refresh(false);
		}
	}
	

	/** 
	 *   Action used when editing an indication structure.
	 *
	 * @author Raquel Hervas
	 * @version 1.0, (Oct. 19, 2009)
	 */
	protected class EditIndicationAction extends EditAction implements ISelectionChangedListener {
	
		/* 
		 * (non-Javadoc) @see edu.mit.story.ui.actions.generic.EditAction#doEdit()
		 */
		@Override
		protected void doEdit() {
			
			ISelection sel = treeViewer.getSelection();
			
			// get description to be edited
			IDesc indDesc = SelectionUtils.extractSingle(sel, IDesc.class);
			if(IndicationStructureRep.getInstance().isType(indDesc)){ 
				editIndication(indDesc, null, null);
				return;
			}
			
			//It can be the ReferentReferencePair
			ReferentReferencePair pair = SelectionUtils.extractSingle(sel, ReferentReferencePair.class);
			if(pair != null){
				editReferent(pair);
				return;
			}
			
			//It can be a list of nucleus or modifiers
			ITypedList list = SelectionUtils.extractSingle(sel, ITypedList.class);
			if(list != null){
				editIndication(findDesc(list.getParent()), null, null);
				return;
			}
				
			INucleus nuc = SelectionUtils.extractSingle(sel, INucleus.class);
			if(nuc != null){
				editIndication(findDesc(nuc.getParent()), nuc, null);
			}
			
			IModifier mod = SelectionUtils.extractSingle(sel, IModifier.class);
			if(nuc != null){
				editIndication(findDesc(nuc.getParent()), null, mod);
				return;
			}
			
		}
		
		protected IDesc findDesc(IIndicationStructure is){
			for(IDesc d : getStoryEditor().getStoryModel().getData().getDescriptions(IndicationStructureRep.getInstance())){
				if(d.getData() == is) return d;
			}
			return null;
		}
		
		protected void editIndication(IDesc indDesc, INucleus nuc, IModifier mod){
			
			// go about getting relation creator view
			IWorkbenchWindow window = getStoryEditor().getEditorSite().getWorkbenchWindow();
			IViewPart part = WorkbenchUtils.openView(window, IStoryUIConstantIDs.VIEW_Creator);
			CreatorView view = SelectionUtils.cast(part, CreatorView.class);
			if(view == null) return;
			
			// get widget
			Control control = view.getCurrentPage().getControl();
			CreatorPageBookWidget widget = SelectionUtils.cast(control, CreatorPageBookWidget.class);
			if(widget == null) return;
			
			// get appropriate creator page
			CreatorPageBookWidget.HelperPageDesc cPage = widget.showPage(IndicationCreatorPage.ID);
			if(cPage == null) return;
			ICreatorPage creator = cPage.getHelper();
			IndicationCreatorPage page = SelectionUtils.cast(creator, IndicationCreatorPage.class);
			if(page == null) return;
			
			// load indication structure into creator
			page.getController().load(indDesc);
			if(nuc != null) page.getController().getNucleusController().load(nuc);
			if(mod != null)	page.getController().getModifierController().load(mod);
		}
		
		protected void editReferent(ReferentReferencePair pair){
			
			// go about getting relation creator view
			IWorkbenchWindow window = getStoryEditor().getEditorSite().getWorkbenchWindow();
			IViewPart part = WorkbenchUtils.openView(window, IStoryUIConstantIDs.VIEW_Creator);
			CreatorView view = SelectionUtils.cast(part, CreatorView.class);
			if(view == null) return;
			
			// get widget
			Control control = view.getCurrentPage().getControl();
			CreatorPageBookWidget widget = SelectionUtils.cast(control, CreatorPageBookWidget.class);
			if(widget == null) return;
			
			// get appropriate creator page
			CreatorPageBookWidget.HelperPageDesc cPage = widget.showPage(IDiscourseUIConstantIDs.CREATOR_Referents);
			if(cPage == null) return;
			ICreatorPage creator = cPage.getHelper();
			ReferentCreatorPage page = SelectionUtils.cast(creator, ReferentCreatorPage.class);
			if(page == null) return;
			
			// load referent into creator
			page.getController().load(pair.getReferentDesc());
			IReferenceModel refModel = page.getController().getReferentModel().getReference(pair.getReference().getID());
			page.getController().getReferenceController().load(refModel);
			
		}
		
		/* 
		 * (non-Javadoc) @see org.eclipse.jface.viewers.ISelectionChangedListener#selectionChanged(org.eclipse.jface.viewers.SelectionChangedEvent)
		 */
		
		public void selectionChanged(SelectionChangedEvent event) {
			//Implement if required
		}
		
	}
	
	/**
	 *   Action for changing the type of nuclei and modifiers of an indication structure
	 *
	 * @author Raquel Hervas
	 * @version 1.0, (Jan. 8, 2010)
	 * @since nil.ucm.indications.ui 1.0.0
	 */
	protected class ChangeIndicationTypeAction extends Action implements ISelectionChangedListener {
		
		private IndicationType newType;
		
		public ChangeIndicationTypeAction(IndicationType type) {
			super();
			StringBuilder sb = new StringBuilder();
			sb.append(type.getDescription());
			sb.insert(1, '&');
			setText(sb.toString());
			treeViewer.addSelectionChangedListener(this);
			newType = type;
		}

		public void run() {
			
			ITiming timing = new Timing(ITiming.SOURCE_USER, true);
//			IIndicationStructure indStruct = null;
			
			//We have selected a modifier
			IModifier mod = SelectionUtils.extractSingle(treeViewer.getSelection(), IModifier.class);
			if (mod != null) {
//				indStruct = mod.getParent();
				
				IDesc desc = null;
				for(IDesc d : getStoryEditor().getStoryModel().getData().getDescriptions(IndicationStructureRep.getInstance())){
					if(!((IIndicationStructure)d.getData()).getModifiers().contains(mod)) continue;
					desc = d;
					break;
				}
				getStoryEditor().getStoryModel().applyChange(this, new ModifyModifierTypeChange(desc, mod, newType, timing), false);
				return;
			}
					
//			IDesc desc = SelectionUtils.extractSingle(treeViewer.getSelection(), IDesc.class);
//			
//			if (desc != null && IndicationStructureRep.getInstance().isType(desc)) {
//				editor.getStoryModel().applyChange(this, new ModifyModifierTypeChange(desc, newType, timing), false);
//			}
			
			//We have selected a nucleus
			INucleus nuc = SelectionUtils.extractSingle(treeViewer.getSelection(), INucleus.class);
			if (nuc != null) {
//				indStruct = nuc.getParent();
				
				IDesc desc = null;
				for(IDesc d : getStoryEditor().getStoryModel().getData().getDescriptions(IndicationStructureRep.getInstance())){
					if(!((IIndicationStructure)d.getData()).getNuclei().contains(nuc)) continue;
					desc = d;
					break;
				}
				getStoryEditor().getStoryModel().applyChange(this, new ModifyNucleusTypeChange(desc, nuc, newType, timing), false);
				return;
			}
		}
		
		/* 
		 * (non-Javadoc) @see org.eclipse.jface.viewers.ISelectionChangedListener#selectionChanged(org.eclipse.jface.viewers.SelectionChangedEvent)
		 */
		public void selectionChanged(SelectionChangedEvent event) {
			
			//We have selected a description
			IDesc desc = SelectionUtils.extractSingle(treeViewer.getSelection(), IDesc.class);
			if (desc != null) {
				setEnabled(false);
				return;
			}
			
			//We have selected the ReferentReferencePair 
			ReferentReferencePair pair = SelectionUtils.extractSingle(treeViewer.getSelection(), ReferentReferencePair.class);
			if (pair != null) {
				setEnabled(false);
				return;
			}
			
			//We have selected a list of nucleus or modifiers
			ITypedList list = SelectionUtils.extractSingle(treeViewer.getSelection(), ITypedList.class);
			if (list != null) {
				setEnabled(false);
				return;
			}
			
			//We have selected a nucleus
			INucleus nuc = SelectionUtils.extractSingle(treeViewer.getSelection(), INucleus.class);
			if (nuc != null){
				setEnabled(true);
				return;				
			}
			
			//We have selected a modifier
			IModifier mod = SelectionUtils.extractSingle(treeViewer.getSelection(), IModifier.class);
			if (mod != null) {
				setEnabled(true);
				return;
			}
			
		}
	}
	
	/** 
	 *   Action for deleting an element in the Indication Structure Details Page
	 *
	 * @author Raquel Hervas
	 * @version 1.0, (Oct. 20, 2009)
	 */
	protected class DeleteElementAction extends DeleteAction implements ISelectionChangedListener {

		public DeleteElementAction() {
			treeViewer.addSelectionChangedListener(this);
		}
	
		/* (non-Javadoc) @see edu.mit.story.ui.actions.generic.DeleteAction#doDelete() */
		
		protected void doDelete() {
			ITiming timing = new Timing(ITiming.SOURCE_USER, true);
			IIndicationStructure indStruct = null;
			
			//We have selected a description
			IDesc desc = SelectionUtils.extractSingle(treeViewer.getSelection(), IDesc.class);
			if (desc != null) {
				getStoryEditor().getStoryModel().applyChange(this, new RemoveDescriptionsChange(desc), false);
				return;
			}
			
			//We have selected the ReferentReferencePair (the complete indication must be deleted)
			ReferentReferencePair pair = SelectionUtils.extractSingle(treeViewer.getSelection(), ReferentReferencePair.class);
			if (pair != null) {
				indStruct = pair.getParent();
				for(IDesc d : getStoryEditor().getStoryModel().getData().getDescriptions(IndicationStructureRep.getInstance())){
					if(d.getData() != indStruct) continue;
					desc = d;
					break;
				}
				getStoryEditor().getStoryModel().applyChange(this, new RemoveDescriptionsChange(desc), false);
				return;
			}
			
			//We have selected a modifier
			IModifier mod = SelectionUtils.extractSingle(treeViewer.getSelection(), IModifier.class);
			if (mod != null) {
				indStruct = mod.getParent();
				
				desc = null;
				for(IDesc d : getStoryEditor().getStoryModel().getData().getDescriptions(IndicationStructureRep.getInstance())){
					if(!((IIndicationStructure)d.getData()).getModifiers().contains(mod)) continue;
					desc = d;
					break;
				}
				getStoryEditor().getStoryModel().applyChange(this, new RemoveModifierChange(desc, mod, timing), false);
				return;
			}
			
			//We have selected a nucleus
			INucleus nuc = SelectionUtils.extractSingle(treeViewer.getSelection(), INucleus.class);
			if (nuc != null) {
				indStruct = nuc.getParent();
				
				desc = null;
				for(IDesc d : getStoryEditor().getStoryModel().getData().getDescriptions(IndicationStructureRep.getInstance())){
					if(!((IIndicationStructure)d.getData()).getNuclei().contains(nuc)) continue;
					desc = d;
					break;
				}
				getStoryEditor().getStoryModel().applyChange(this, new RemoveNucleusChange(desc, nuc, timing), false);
				return;
			}
		}
	
		/* 
		 * (non-Javadoc) @see org.eclipse.jface.viewers.ISelectionChangedListener#selectionChanged(org.eclipse.jface.viewers.SelectionChangedEvent)
		 */
		
		public void selectionChanged(SelectionChangedEvent event) {
			
			//We have selected a description
			IDesc desc = SelectionUtils.extractSingle(treeViewer.getSelection(), IDesc.class);
			if (desc != null) {
				setEnabled(true);
				return;
			}
			
			//We have selected the ReferentReferencePair 
			ReferentReferencePair pair = SelectionUtils.extractSingle(treeViewer.getSelection(), ReferentReferencePair.class);
			if (pair != null) {
				setEnabled(true);
				return;
			}
			
			//We have selected a list of nucleus or modifiers
			ITypedList list = SelectionUtils.extractSingle(treeViewer.getSelection(), ITypedList.class);
			if (list != null) {
				setEnabled(false);
				return;
			}
			
			//We have selected a nucleus
			INucleus nuc = SelectionUtils.extractSingle(treeViewer.getSelection(), INucleus.class);
			if (nuc != null){
				setEnabled(nuc.getParent().getNuclei().size() > 1);
				return;				
			}
			
			//We have selected a modifier
			IModifier mod = SelectionUtils.extractSingle(treeViewer.getSelection(), IModifier.class);
			if (mod != null) {
				setEnabled(true);
				return;
			}
			
		}

	}

	/**
		 *   Action for changing the type of nuclei and modifiers of an indication structure
		 *
		 * @author Raquel Hervas
		 * @version 1.0, (Jan. 8, 2010)
		 * @since nil.ucm.indications.ui 1.0.0
		 */
		protected class ToggleCopularAction extends Action implements ISelectionChangedListener {
			
			public ToggleCopularAction() {
				super();
				setText("&Toggle Copular");
				treeViewer.addSelectionChangedListener(this);
			}
	
			public void run() {
				
				ITiming timing = new Timing(ITiming.SOURCE_USER, true);
				
				// get description to be edited
				IDesc desc = SelectionUtils.extractSingle(treeViewer.getSelection(), IDesc.class);
				if(desc == null){
					ReferentReferencePair pair = SelectionUtils.extractSingle(treeViewer.getSelection(), ReferentReferencePair.class);
					if(pair != null){
						// find the referent desc
						for(IDesc d : getStoryEditor().getStoryModel().getData().getDescriptions(IndicationStructureRep.getInstance())){
							if(d.getData() != pair.getParent()) continue;
							desc = d;
							break;
						}
					}
				}
				
				if(desc == null) return;
				
				getStoryEditor().getStoryModel().applyChange(this, new SetCopularPredicateChange(desc, timing), false);
			}
			
			/* 
			 * (non-Javadoc) @see org.eclipse.jface.viewers.ISelectionChangedListener#selectionChanged(org.eclipse.jface.viewers.SelectionChangedEvent)
			 */
			public void selectionChanged(SelectionChangedEvent event) {
				
				//We have selected a description
				IDesc desc = SelectionUtils.extractSingle(treeViewer.getSelection(), IDesc.class);
				if (desc != null) {
					setEnabled(true);
					return;
				}
				
				//We have selected the ReferentReferencePair 
				ReferentReferencePair pair = SelectionUtils.extractSingle(treeViewer.getSelection(), ReferentReferencePair.class);
				if (pair != null) {
					setEnabled(true);
					return;
				}
				
				//We have selected a list of nucleus or modifiers
				ITypedList list = SelectionUtils.extractSingle(treeViewer.getSelection(), ITypedList.class);
				if (list != null) {
					setEnabled(false);
					return;
				}
				
				//We have selected a nucleus
				INucleus nuc = SelectionUtils.extractSingle(treeViewer.getSelection(), INucleus.class);
				if (nuc != null){
					setEnabled(false);
					return;				
				}
				
				//We have selected a modifier
				IModifier mod = SelectionUtils.extractSingle(treeViewer.getSelection(), IModifier.class);
				if (mod != null) {
					setEnabled(false);
					return;
				}
				
			}
		}

}
