/* Filename: ArgumentEntryWiget.java
 * Author: Mark A. Finlayson
 * Format: Java 2 v1.5.0
 * Date created: Oct 12, 2007
 */
package nil.ucm.indications2.ui.widgets;



import java.util.Arrays;

import nil.ucm.indications2.core.rep.model.IReferenceModel;
import nil.ucm.indications2.ui.controllers.IReferenceController;

import org.eclipse.jface.action.IAction;
import org.eclipse.jface.viewers.StyledString;
import org.eclipse.jface.viewers.DelegatingStyledCellLabelProvider.IStyledLabelProvider;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;

import edu.mit.discourse.core.rep.referent.IReference;
import edu.mit.discourse.core.rep.referent.RefPair;
import edu.mit.discourse.ui.label.ReferentLabelProvider;
import edu.mit.story.core.desc.IDesc;
import edu.mit.story.ui.widgets.AnnotationWidget;
import edu.mit.story.ui.widgets.ButtonAction;

/** 
 * An annotation widget responsible for capturing a single token from a story editor.
 *
 * @author M.A. Finlayson
 * @version $Rev$, $LastChangedDate$
 * @since edu.mit.parsing.ui 1.0.0
 */
public class ReferenceCaptureWidget extends AnnotationWidget<IReferenceController> {
	
	protected final IStyledLabelProvider lp;
	
	// UI controls
	protected StyledText tokenText;
	protected Button captureButton;
	protected Button clearButton;
	protected ButtonAction captureAction;
	protected ButtonAction clearAction;
	/**
	 * Creates a new desc capture widget on the specified parent, with the
	 * specified style, using annotations with the specified annotation id.
	 * Annotations governed by this widget are shown in the specified story
	 * editor.
	 * 
	 * The style value is either one of the style constants defined in class
	 * <code>SWT</code> which is applicable to instances of this class, or must
	 * be built by <em>bitwise OR</em>'ing together (that is, using the
	 * <code>int</code> "|" operator) two or more of those <code>SWT</code>
	 * style constants. The class description lists the style constants that are
	 * applicable to the class. Style bits are also inherited from superclasses.
	 * </p>
	 * 
	 * @param parent
	 *            a widget which will be the parent of the new instance (cannot
	 *            be null)
	 * @param style
	 *            the style of widget to construct
	 * @param annoID
	 *            the id of annotations that will be added by this widget
	 * @param editor
	 *            the story editor in which annotations will be shown, and from
	 *            which token captures will be made
	 * @param defaultMessage
	 *            the default message delivered by this widget
	 * @throws NullPointerException
	 *             if the given editor or annotation id is <code>null</code>
	 * @throws IllegalArgumentException
	 *             if the annotation id is empty or all whitespace
	 * @since edu.mit.parsing.ui 1.0.0
	 */
	public ReferenceCaptureWidget(Composite parent, int style, IReferenceController controller) {
		super(parent, style, controller);
		controller.getModel().addChangeListener(this);
		this.lp = new ReferentLabelProvider(controller.getStoryEditor());
	}
	
	/**
	 * TODO: Write comment
	 *
	 * @since edu.mit.parsing.ui 1.0.0
	 */
	public IAction getCaptureAction(){
		return captureAction;
	}
	
	/**
	 * TODO: Write comment
	 *
	 * @since edu.mit.parsing.ui 1.0.0
	 */
	public IAction getClearAction(){
		return clearAction;
	}

	/* 
	 * (non-Javadoc) @see edu.mit.story.ui.widgets.AnnotationWidget#createControl(org.eclipse.swt.widgets.Composite)
	 */
	protected void createControl(Composite parent){

		GridData gd;
		
		// add color and layout to parent
		GridLayout layout = new GridLayout(5, false);
		layout.marginHeight = 0;
		layout.marginWidth = 0;
		parent.setLayout(layout);

		// create token text control
		tokenText = new StyledText(parent, SWT.BORDER);
		tokenText.setBackground(parent.getBackground());
		tokenText.setForeground(parent.getForeground());
		tokenText.setEditable(false);
		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.minimumWidth = 30; // FIXME: arbitrary minimum width
		tokenText.setLayoutData(gd);

		// create capture button
		captureButton = new Button(parent, SWT.NONE);
		captureButton.setText("Capture");
		captureButton.setToolTipText("Capture token from the editor");
		captureAction = new ButtonAction(captureButton){
			public void run() {
				controller.capture();
			}
		};
		captureAction.setText("Capture Token");
		captureAction.setToolTipText(captureButton.getToolTipText());
		
		// additional controls for subclasses
		Composite extrasParent = new Composite(parent, SWT.NONE);
		extrasParent.setBackground(parent.getBackground());
		extrasParent.setForeground(parent.getForeground());
		extrasParent.setLayoutData(new GridData(SWT.FILL, SWT.NONE, false, false));
		createAdditionalControls(extrasParent);
		
		// clear button
		clearButton = new Button(parent, SWT.NONE);
		clearButton.setText("Clear");
		clearButton.setToolTipText("Clear field");
		clearAction = new ButtonAction(clearButton){
			public void run() {
				controller.clear();
			}
		};
		clearAction.setText("Clear Token");
		clearAction.setToolTipText(clearButton.getToolTipText());

		// clear and annotate button
		createAnnotateButton(parent);
	}
	
	/**
	 * TODO: Write comment
	 *
	 * @param parent
	 * @since edu.mit.parsing.ui 1.0.0
	 */
	protected void createAdditionalControls(Composite parent){
		GridData gd = (GridData)parent.getLayoutData();
		gd.widthHint = 0;
		parent.setLayout(new GridLayout());
	}
	
	/* 
	 * (non-Javadoc) @see edu.mit.story.ui.widgets.AnnotationWidget#updateFromModel()
	 */
	protected void updateFromModel(){
		
		// model parameters
		IReferenceModel model = controller.getModel();
		IReference ref = model.getReference();
		IDesc desc = model.getReferentDescription();
		
		// calculate what state the view should be in
		StyledString displayText = ref == null ? new StyledString() : lp.getStyledText(new RefPair(desc, ref));
		boolean clearEnabled = !model.isCleared();
		
		// test controls to see if they're in that state; if not, set them
		if(!tokenText.getText().equals(displayText.getString()))  tokenText.setText(displayText.getString());
		if(!Arrays.equals(tokenText.getStyleRanges(), displayText.getStyleRanges())) tokenText.setStyleRanges(displayText.getStyleRanges());
		if(clearButton.isEnabled() != clearEnabled) clearButton.setEnabled(clearEnabled);
	}

}
