/* Filename: LexicalMarker.java
 * Author: M. A. Finlayson
 * Format: Java 2 v1.5.0
 * Date created: Dec 6, 2007
 */
package edu.mit.discourse.core.rep.relation;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import edu.mit.story.core.desc.IDesc;
import edu.mit.story.core.desc.ISegment;
import edu.mit.story.core.mappers.BlockMapper;
import edu.mit.story.core.model.IStoryData;
import edu.mit.story.core.model.IStoryModel;
import edu.mit.story.core.position.HasPositionSet;
import edu.mit.story.core.position.IHasPosition;
import edu.mit.story.core.position.IHasPositionSet;
import edu.mit.story.core.position.SimplePosition;

/** One line description goes here...
 * More detail goes here...
 *
 * @author M.A. Finlayson
 * @since 1.5.0
 */
public class LexicalMarker extends Argument implements ILexicalMarker {

	private final String implicitMarker;
	private final Type type;

	/**
	 * Constructs a implicit, null lexical marker.
	 */
	public LexicalMarker(Collection<? extends IAdjunct> adjuncts){
		super(null, adjuncts);
		this.type = Type.NULL;
		this.implicitMarker = null;
	}
	
	/**
	 * Constructs an implicit lexical marker with the specified string
	 * as marker.
	 */
	public LexicalMarker(String lexicalMarker, Collection<? extends IAdjunct> adjuncts){
		super(null, adjuncts);
		if(lexicalMarker.trim().length() == 0) throw new IllegalArgumentException();
		this.type = Type.IMPLICIT;
		this.implicitMarker = lexicalMarker;
	}
	
	public LexicalMarker(Collection<? extends ISegment> segments, Collection<? extends IAdjunct> adjuncts){
		super(segments, adjuncts);
		this.type = Type.EXPLICIT;
		this.implicitMarker = null;
	}
	
	
	public String generateText(){
		if(isNull()){
			return ILexicalMarker.nullDisplay;
		} else if(isImplicit()){
			return getImplicitMarker() + ILexicalMarker.implicitDisplay;
		} else {
			return super.generateText();
		}
	}
	
	
	/* (non-Javadoc) @see edu.mit.discourse.core.rep.relation.ILexicalMarker#getImplicitMarker() */
	public String getImplicitMarker(){
		return implicitMarker;
	}
	
	public Type getType(){
		return type;
	}
	
	/* (non-Javadoc) @see edu.mit.discourse.core.rep.relation.ILexicalMarker#isExplicit() */
	public boolean isExplicit() {
		return type == Type.EXPLICIT;
	}

	/* (non-Javadoc) @see edu.mit.discourse.core.rep.relation.ILexicalMarker#isImplicit() */
	public boolean isImplicit() {
		return type == Type.IMPLICIT;
	}

	/* (non-Javadoc) @see edu.mit.discourse.core.rep.relation.ILexicalMarker#isNull() */
	public boolean isNull() {
		return type == Type.NULL;
	}
	
	/* (non-Javadoc) @see edu.mit.story.core.desc.IStructuredData#calculatePosition() */
	public IHasPosition calculatePosition() {
		if(isExplicit()){
			return super.calculatePosition();
		} else {
			IHasPositionSet<IHasPosition> result = new HasPositionSet<IHasPosition>();
			for(IAdjunct adjunct : getAdjuncts()) result.add(adjunct.getSegments());
			return result.isEmpty() ? null : new SimplePosition(result);
		}
	}
	
	@Override
	public Argument recalculate(IDesc container, IStoryModel provider) {
		
		Argument newInstance = super.recalculate(container, provider);
		if(newInstance == null) return null;
		if(newInstance == this) return this;
		
		if(isNull()){
			return new LexicalMarker(newInstance.getAdjuncts());
		} else if(isImplicit()){
			return new LexicalMarker(implicitMarker, newInstance.getAdjuncts());
		} else {
			return new LexicalMarker(getSegments(), newInstance.getAdjuncts());
		}
	}

	@Override
	public LexicalMarker clone() {
		if(isNull()){
			return new LexicalMarker(getAdjuncts());
		} else if(isImplicit()){
			return new LexicalMarker(implicitMarker, getAdjuncts());
		} else {
			return new LexicalMarker(getSegments(), getAdjuncts());
		}
	}

	public static String serialize(ILexicalMarker lm){
		if(lm.getType() == Type.NULL){
			String[] blocks = new String[lm.getAdjuncts().size()+1];
			blocks[0] = ILexicalMarker.nullDisplay;
			int i = 1;
			for(IAdjunct adjunct : lm.getAdjuncts()) blocks[i++] = Adjunct.serialize(adjunct);
			return BlockMapper.SLASH_FORWARD.serialize(Arrays.asList(blocks));
		} else if (lm.getType() == Type.IMPLICIT){
			String[] blocks = new String[lm.getAdjuncts().size()+1];
			blocks[0] = ILexicalMarker.prefixImplicit + lm.getImplicitMarker();
			int i = 1;
			for(IAdjunct adjunct : lm.getAdjuncts()) blocks[i++] = Adjunct.serialize(adjunct);
			return BlockMapper.SLASH_FORWARD.serialize(Arrays.asList(blocks));
		} else {
			return Argument.serialize(lm);
		}
	}
	
	public static LexicalMarker reconstitute(String description, IStoryData data){
		List<String> blocks = BlockMapper.SLASH_FORWARD.reconstitute(description);
		
		String first = blocks.get(0);
		
		// null lexical marker
		if(first.startsWith(ILexicalMarker.nullDisplay)){
			if(blocks.isEmpty()){
				return new LexicalMarker(null);
			} else {
				// extract adjuncts
				List<IAdjunct> adjuncts = new ArrayList<IAdjunct>(blocks.size());
				IAdjunct adjunct;
				for(int i = 0; i < blocks.size()-1; i++){
					adjunct = Adjunct.reconstitute(blocks.get(i+1), data);
					if(adjunct != null) adjuncts.add(adjunct);
				}
				return new LexicalMarker(adjuncts);
			}
		}
		
		// implicit lexical marker
		if(first.startsWith(ILexicalMarker.prefixImplicit)){ // implicit
			String marker = first.substring(ILexicalMarker.prefixImplicit.length());
			if(blocks.isEmpty()){
				return new LexicalMarker(marker, null);
			} else {
				// extract adjuncts
				List<IAdjunct> adjuncts = new ArrayList<IAdjunct>(blocks.size());
				IAdjunct adjunct;
				for(int i = 0; i < blocks.size()-1; i++){
					adjunct = Adjunct.reconstitute(blocks.get(i+1), data);
					if(adjunct != null) adjuncts.add(adjunct);
				}
				return new LexicalMarker(marker, adjuncts);
			}
		}
		
		// explicit lexical marker
		Argument arg = Argument.reconstitute(description, data);
		return new LexicalMarker(arg.getSegments(), arg.getAdjuncts());
	}
	



	

	
}
