/* Filename: IndicationStructureModel.java
 * Creator: Raquel Hervas
 * Format: Java 2 v1.6.0
 * Date created: 30/09/2009
 */
package nil.ucm.indications2.core.rep.model;

import java.util.ArrayList;
import java.util.List;

import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import nil.ucm.indications2.core.rep.IIndicationStructure;
import nil.ucm.indications2.core.rep.IModifier;
import nil.ucm.indications2.core.rep.INucleus;
import nil.ucm.indications2.core.rep.IndicationStructure;
import nil.ucm.indications2.core.rep.IndicationStructureRep;
import nil.ucm.indications2.core.rep.Modifier;
import nil.ucm.indications2.core.rep.Nucleus;

import org.eclipse.jface.dialogs.IMessageProvider;

import edu.mit.discourse.core.rep.referent.IReference;
import edu.mit.story.core.align.IAlignedStoryModel;
import edu.mit.story.core.datamodel.AbstractViewModel;
import edu.mit.story.core.desc.IData;
import edu.mit.story.core.desc.IDesc;
import edu.mit.story.core.desc.IStructuredData;
import edu.mit.story.core.model.IStoryModel;
import edu.mit.story.core.model.change.IModelListener;
import edu.mit.story.core.model.change.StoryChangeEvent;
import edu.mit.story.core.notify.MessageProviderManager;
import edu.mit.story.core.position.IHasPosition;
import edu.mit.story.core.position.PositionUtils;
import edu.mit.story.core.validation.Message;

/** 
 *   Implementation for the indication structure model.
 *
 * @author Raquel Hervas
 * @version 1.0, (Jan. 10, 2010)
 * @since nil.ucm.indications.ui 1.0.0
 */
public class IndicationStructureModel extends AbstractViewModel implements
		IIndicationStructureModel, ChangeListener, IModelListener {

	public static final IMessageProvider noReference = new Message("The indication must have at least one reference", IMessageProvider.ERROR);
	public static final IMessageProvider noNuclei = new Message("The indication must have at least one nucleus", IMessageProvider.ERROR);
	public static final IMessageProvider deletedReferent = new Message("The selected referent does not exist", IMessageProvider.ERROR);
	public static final IMessageProvider deletedReference = new Message("The selected reference does not exist", IMessageProvider.ERROR);
	
	private final IStoryModel model;

	//Internal data for the indication structure
	private final IReferenceModel refModel;
	private final List<INucleusModel> nuclei = new ArrayList<INucleusModel>();
	private final List<IModifierModel> modifiers = new ArrayList<IModifierModel>();
	private boolean isCopularPredicate = false;
	private IDesc loaded = null;
	private MessageProviderManager manager;
	
	public IndicationStructureModel(IStoryModel model) {
		
		// story model
		if(model == null) throw new NullPointerException();
		this.model = model;
		this.model.addModelListener(this);	

		this.refModel = new ReferenceModel(model);
		
		// message manager
		this.manager = new MessageProviderManager();
		this.manager.addChangeListener(this);
		this.manager.add(refModel);
	}
	
	/* 
	 * (non-Javadoc) @see nil.ucm.indications.core.rep.IIndicationStructure#getName()
	 */
	
	public String getName() {
		if (refModel != null) 
			return refModel.getDisplayText();
		else
			return "";
	}

	/* 
	 * (non-Javadoc) @see nil.ucm.indications2.ui.models.IIndicationStructureModel#getReference()
	 */
	public IReferenceModel getReference() {
		return refModel;
	}

	/* 
	 * (non-Javadoc) @see nil.ucm.indications2.core.rep.IIndicationStructure#getReferentDescription()
	 */
	public IDesc getReferentDescription() {
		return refModel.getReferentDescription();
	}

	public List<IModifierModel> getModifiers() {
		return modifiers;
	}

	
	public List<INucleusModel> getNuclei() {
		return nuclei;
	}
	
	
	/* 
	 * (non-Javadoc) @see nil.ucm.indications2.core.rep.IIndicationStructure#isCopularPredicate()
	 */
	public boolean isCopularPredicate() {
		return isCopularPredicate;
	}

	/* 
	 * (non-Javadoc) @see nil.ucm.indications2.ui.models.IIndicationStructureModel#setCopularPredicate(boolean)
	 */
	public void setCopularPredicate(boolean value) {
		if(isCopularPredicate == value) return;
		isCopularPredicate = value;
		setOutOfSync();
		notifyChangeListeners();
	}

	public void addNucleus(INucleusModel nuc/*, IndicationType type*/) {
		
		if (!nuclei.contains(nuc)) {
			nuc.setParent(this);
			if(nuclei.add(nuc)) nuc.addChangeListener(this);
		}
		setOutOfSync();
		notifyChangeListeners();
	}
	
	
	public void addModifier(IModifierModel mod) {
		
		if (!modifiers.contains(mod)) {
			mod.setParent(this);
			if(modifiers.add(mod)) mod.addChangeListener(this);
		}
		setOutOfSync();
		notifyChangeListeners();
	}
	
	
//	public void addReferent(IDesc ref) {		
//		this.referentDesc = ref;
//	}
	
	public void removeModifier(IModifierModel mod) {
		if(modifiers.remove(mod)) mod.removeChangeListener(this);
		
		setOutOfSync();
		notifyChangeListeners();
	}
	
	
	public void removeNucleus(INucleusModel nuc) {
		if (nuclei.remove(nuc)) nuc.removeChangeListener(this);
		
		setOutOfSync();
		notifyChangeListeners();
	}
	
	
	protected IMessageProvider calculateMessage() {
		if(nuclei.isEmpty()) return noNuclei;
		
		IReference r = refModel.getReference();
		if(r != null){
			for(INucleusModel nuc : nuclei){
				if(!PositionUtils.contains(r, nuc.calculatePosition()))
					return new Message("The nucleus '" + nuc.getDisplayText() + "' lies at least partially outside the bounds of the reference", ERROR);
			}
			for(IModifierModel mod : modifiers){
				if(!PositionUtils.contains(r, mod.calculatePosition()))
					return new Message("The modifier '" + mod.getDisplayText() + "' lies at least partially outside the bounds of the reference", ERROR);
			}
		}
		
		return manager.getFixedMessage();
	}

	/* 
	 * (non-Javadoc) @see edu.mit.story.core.position.IHasPosition#getLength()
	 */
	
	public int getLength() {
		return refModel.getLength();
	}

	/* 
	 * (non-Javadoc) @see edu.mit.story.core.position.IHasPosition#getOffset()
	 */
	
	public int getOffset() {
		return refModel.getOffset();
	}

	/* 
	 * (non-Javadoc) @see edu.mit.story.core.position.IHasPosition#getRightOffset()
	 */
	
	public int getRightOffset() {
		return refModel.getRightOffset();
	}

	/* 
	 * (non-Javadoc) @see edu.mit.story.core.desc.IStructuredData#calculatePosition()
	 */
	
	public IHasPosition calculatePosition() {
		return PositionUtils.makePosition(refModel);
	}

	/* 
	 * (non-Javadoc) @see edu.mit.story.core.desc.IStructuredData#getDisplayPosition()
	 */
	
	public IHasPosition getDisplayPosition() {
		return calculatePosition();
	}

	/* 
	 * (non-Javadoc) @see edu.mit.story.core.desc.IStructuredData#recalculate(edu.mit.story.core.model.IStoryModel)
	 */
	
	public IStructuredData recalculate(IDesc container, IStoryModel model) {
		throw new UnsupportedOperationException();
	}
	
	/* 
	 * (non-Javadoc) @see edu.mit.story.core.desc.IData#equals(edu.mit.story.core.desc.IData, edu.mit.story.core.align.IAlignedStoryModel)
	 */
	public boolean equals(IData tgtData, IAlignedStoryModel model) {
		throw new UnsupportedOperationException();
	}

	/* 
	 * (non-Javadoc) @see edu.mit.story.core.util.IClearable#isCleared()
	 */
	
	public boolean isCleared() {
		return loaded == null && !isCopularPredicate && nuclei.isEmpty() && modifiers.isEmpty() && refModel.isCleared();
	}
	
	protected void internalClear(){
		refModel.clear();
		isCopularPredicate = false;
		//Remove the nuclei and modifiers one by one
		for (INucleusModel n : nuclei) {
			n.removeChangeListener(this);
			n.dispose();
		}
		
		for (IModifierModel m :  modifiers) {
			m.removeChangeListener(this);
			m.dispose();
		}
		
		nuclei.clear();
		modifiers.clear();
		
		loaded = null;
	}

	/* 
	 * (non-Javadoc) @see edu.mit.story.core.util.IDisposable#dispose()
	 */
	
	public void dispose() {
		this.model.removeModelListener(this);
	}

	/* 
	 * (non-Javadoc) @see edu.mit.story.core.model.change.IModelListener#modelChanged(edu.mit.story.core.model.change.StoryChangeEvent)
	 */
	public void modelChanged(StoryChangeEvent e) {
		if(loaded == null) return;
		if(!e.affects(IndicationStructureRep.getInstance())) return;
		IDesc newDesc = e.getModel().getData().getDescriptions(IndicationStructureRep.getInstance()).getDescription(loaded.getID());
		if(newDesc == null) clear();
		
//		//If the description changed was a referent
//		if(e.getChange(ReferentRep.getInstance()) != null && referentDesc != null) {
//			
//			//We check if the referent of the indication exists. If not, the indication must be deleted
//			if (model.getData().getDescription(referentDesc.getID()) == null)
//			{
//				clear();
//				return;
//			}
//			
//			//We also have to check if a reference has been deleted.
//			//When a reference is deleted, the referent description is deleted and created again
//			Change c = e.getChange(ReferentRep.getInstance());
//			IHasPositionSet<IDesc> changedRefs = c.getChanges(Flag.DESC_ADDED);
//			
//			boolean referenceDeleted = true;
//			for (IDesc ref : changedRefs)
//			{
//				if (((IReferent)ref.getData()).getReference(refModel.getID()) != null)
//					referenceDeleted = false;
//			}
//			
//			if (referenceDeleted)
//			{
//				clear();
//				return;
//			}
//		}
//		
//		if(loaded == null) return;
//		
//		//If the description changed was an indication structure
//		if(e.getChange(IndicationStructureRep.getInstance()) != null) {
//			
//			IDesc newDesc = model.getData().getDescription(loaded.getID());
//			
//			// description is unchanged
//			if(newDesc == loaded) return;
//			
//			// description was deleted
//			if(newDesc == null){
//				clear();
//				return;
//			} 
//			
//			// description needs to be reloaded
//			load(newDesc);
//		}
	}

		
	
	public IDesc getLoaded() {
		return loaded;
	}


	
	public void load(IDesc desc) {
		
		internalClear();
		
		IIndicationStructure indication = (IIndicationStructure)desc.getData();
		
		isCopularPredicate = indication.isCopularPredicate();

		//We load the referent
		refModel.setReference(indication.getReferentDescription(), indication.getReference().getID());
		
		//We load the nuclei and modifiers
		INucleusModel nucModel;
		for (INucleus nuc : indication.getNuclei()){
			nucModel = new NucleusModel(model);
			nucModel.load(nuc);
			nuclei.add(nucModel);
		}
		
		IModifierModel modModel;
		for (IModifier mod : indication.getModifiers()){
			modModel = new ModifierModel(model);
			modModel.load(mod);
			modifiers.add(modModel);
		}
		
		//We load the type
		this.loaded = desc;
		setOutOfSync();
		notifyChangeListeners();
	}

	/* 
	 * (non-Javadoc) @see javax.swing.event.ChangeListener#stateChanged(javax.swing.event.ChangeEvent)
	 */
	public void stateChanged(ChangeEvent e) {
		setOutOfSync();
		notifyChangeListeners();
	}

	public IIndicationStructure toIndicationStructure() {
			
			// Create nuclei
			List<INucleus> nucleiList = new ArrayList<INucleus>(nuclei.size());
			for (INucleusModel nuc : nuclei) {
				nucleiList.add(new Nucleus(nuc.getSegments(), nuc.getType()));
			}
			
			// Create modifers
			List<IModifier> modList = new ArrayList<IModifier>(modifiers.size());
			for (IModifierModel mod : modifiers) {
				modList.add(new Modifier(mod.getSegments(),mod.getType()));
			}
			
			IIndicationStructure indStruct = new IndicationStructure(refModel.getReferentDescription(), refModel.getReference().getID(), nucleiList, modList, isCopularPredicate);
			return indStruct;
		}

}
