/* Filename: IndicationStructureListContentProvider.java
 * Creator: Raquel Hervas
 * Format: Java 2 v1.6.0
 * Date created: 23/09/2009
 */
package nil.ucm.indications2.ui.content;

import java.util.ArrayList;
import java.util.List;

import nil.ucm.indications2.core.rep.IIndicationStructure;
import nil.ucm.indications2.core.rep.IndicationStructureRep;
import nil.ucm.indications2.core.rep.model.IIndicationStructureModel;
import nil.ucm.indications2.ui.IndicationsUIMessages;
import edu.mit.discourse.core.rep.referent.IReference;
import edu.mit.story.core.desc.IDesc;
import edu.mit.story.ui.content.StoryEditorContentProvider;
import edu.mit.story.ui.editor.IStoryEditor;

/** 
 * TODO: Write comment
 *
 * @author Raquel Hervas
 * @version 1.0, (Jan. 05, 2010)
 * @since nil.ucm.indications.ui 1.0.0
 */
public class IndicationStructureContentProvider extends StoryEditorContentProvider {
	
	/**
	 * Constructs a new content provider with no input
	 */
	public IndicationStructureContentProvider(){
		this(null);
	}
	
	public IndicationStructureContentProvider(IStoryEditor input){
		super(IndicationStructureRep.getInstance(), input, IndicationsUIMessages.MSG_NoIndicationStructures);
	}
	
	public boolean hasChildren(Object element) {
		
		//Indication structures always have children (at least nuclei and modifiers)
		if(element instanceof IDesc) {
			IDesc desc = (IDesc)element;
			if(IndicationStructureRep.getInstance().isType(desc)) return true;
		}
		
		//Handle indication structure model. Indication structures always have children,
		// but it is possible that they are not loaded
		if(element instanceof IIndicationStructureModel){
			return true;
		}
		
		//If this is a list (of nuclei or modifiers), it has children if it has elements
		if(element instanceof ITypedList){
			return ((ITypedList)element).getInternalList().size() > 0;
		}
		
		// send to super
		return super.hasChildren(element);
	}
	
	public Object[] getChildren(Object parent) {
		
		//If element is a indication structure description, extract it's data for the next if block
		if(parent instanceof IDesc){
			IDesc desc = (IDesc)parent;
			if(IndicationStructureRep.getInstance().isType(desc)) parent = desc.getData();  //When the element is the description of the indication
		}
		
		//Children of an indication structure are the associated referent/reference pair, and the nuclei with the list of modifiers
		if(parent instanceof IIndicationStructure){
			IIndicationStructure indStruct = (IIndicationStructure)parent;
			
			//Allocate children array
			List<Object> result = new ArrayList<Object>(3);
			//List with a string with the list of nuclei, and the list of modifiers
			IDesc desc = indStruct.getReferentDescription();
			IReference ref = indStruct.getReference();
			if(desc != null && ref != null) result.add(new ReferentReferencePair(desc, ref, indStruct));
			result.add(new TypedNucleusList(indStruct.getNuclei(),indStruct));
			if(!indStruct.getModifiers().isEmpty()) result.add(new TypedModifierList(indStruct.getModifiers(),indStruct));
			return result.toArray();
		}
		
		//Children of a list are its children
		if(parent instanceof ITypedList){
			return ((ITypedList)parent).getInternalList().toArray();
		}
		
		//Defer to super as always
		return super.getChildren(parent);
	}
	
	/**
	 * 
	 *   Internal class required to show information about both the referent and the referent in the
	 * same line of the Details View
	 *
	 * @author Raquel Hervas
	 * @version 1.0, (Sep. 29, 2009)
	 * @since nil.ucm.indications.ui 1.0.0
	 */
	public class ReferentReferencePair {
		
		private IDesc referentDesc;
		private IReference reference;
		private IIndicationStructure parent;
		
		public ReferentReferencePair(IDesc refDesc, IReference ref, IIndicationStructure p){
			if(refDesc == null) throw new NullPointerException();
			if(ref == null) throw new NullPointerException();
			if(p == null) throw new NullPointerException();
			this.reference = ref;
			this.referentDesc = refDesc;
			this.parent = p;
		}
		
		public IDesc getReferentDesc() {
			return referentDesc;
		}

		public IReference getReference() {
			return reference;
		}
		
		public IIndicationStructure getParent() {
			return parent;
		}

		/* 
		 * (non-Javadoc) @see java.lang.Object#hashCode()
		 */
		@Override
		public int hashCode() {
			int result = 31 + reference.getID();
			return 31 * result + (int)referentDesc.getID();
		}

		/* 
		 * (non-Javadoc) @see java.lang.Object#equals(java.lang.Object)
		 */
		@Override
		public boolean equals(Object obj) {
			if (this == obj) return true;
			if (obj == null) return false;
			if (getClass() != obj.getClass()) return false;
			ReferentReferencePair other = (ReferentReferencePair) obj;
			if (reference.getID() != other.reference.getID()) return false;
			if (referentDesc.getID() != other.referentDesc.getID()) return false;
			return true;
		}

	}
	
	/**
	 * 
	 *   This interface is a wrapper for lists containing INucleus or IModifier.
	 *
	 * @author Raquel Hervas
	 */
	public interface ITypedList {
		
		public List<?> getInternalList();
		
		public IIndicationStructure getParent();
		
	}
	
	/**
	 * 
	 *   This class is a wrapper for lists containing INucleus. With this, it is possible to
	 * distinguish correctly between INucleus and IModifier lists in the LabelProvider.
	 * 
	 * @author Raquel Hervas
	 */
	public class TypedNucleusList implements ITypedList {
		
		private List<?> internalList;
		private final IIndicationStructure parent;
		
		public TypedNucleusList(List<?> l, IIndicationStructure p) {
			internalList = l;
			parent = p;
		}

		public List<?> getInternalList() {
			return internalList;
		}

		public IIndicationStructure getParent() {
			return parent;
		}

		/* 
		 * (non-Javadoc) @see java.lang.Object#hashCode()
		 */
		@Override
		public int hashCode() {
			int prime = 31;
			IDesc desc = parent.getReferentDescription();
			if(desc != null){
				return prime + (int)desc.getID();
			} else {
				return prime + parent.hashCode();
			}
		}

		/* 
		 * (non-Javadoc) @see java.lang.Object#equals(java.lang.Object)
		 */
		@Override
		public boolean equals(Object obj) {
			if (this == obj) return true;
			if (obj == null) return false;
			if (getClass() != obj.getClass()) return false;
			TypedNucleusList other = (TypedNucleusList) obj;
			
			IDesc me = parent.getReferentDescription();
			IDesc it = other.getParent().getReferentDescription();
			if(me != null && it != null && me.getID() != it.getID()) return false;
			if(parent.getReference().getID() != other.getParent().getReference().getID()) return false;
			return true;
		}
	}
	
	/**
	 * 
	 *   This class is a wrapper for lists containing IModifier. With this, it is possible to
	 * distinguish correctly between INucleus and IModifier lists in the LabelProvider.
	 *
	 * @author Raquel Hervas
	 */
	public class TypedModifierList implements ITypedList {
		
		private List<?> internalList;
		private IIndicationStructure parent;
		
		public TypedModifierList(List<?> l, IIndicationStructure p) {
			internalList = l;
			parent = p;
		}

		public List<?> getInternalList() {
			return internalList;
		}
		
		public IIndicationStructure getParent() {
			return parent;
		}

		/* 
		 * (non-Javadoc) @see java.lang.Object#hashCode()
		 */
		@Override
		public int hashCode() {
			int prime = 31;
			IDesc desc = parent.getReferentDescription();
			if(desc != null){
				return prime + (int)desc.getID();
			} else {
				return prime + parent.hashCode();
			}
		}

		/* 
		 * (non-Javadoc) @see java.lang.Object#equals(java.lang.Object)
		 */
		@Override
		public boolean equals(Object obj) {
			if (this == obj) return true;
			if (obj == null) return false;
			if (getClass() != obj.getClass()) return false;
			TypedModifierList other = (TypedModifierList) obj;
			
			IDesc me = parent.getReferentDescription();
			IDesc it = other.getParent().getReferentDescription();
			if(me != null && it != null && me.getID() != it.getID()) return false;
			if(parent.getReference().getID() != other.getParent().getReference().getID()) return false;
			return true;
		}
	}

}
