/* Filename: ACLExportHandler.java
 * Creator: M.A. Finlayson
 * Format: Java 2 v1.6.0
 * Date created: May 9, 2010
 */
package nil.ucm.indications2.ui.handlers;

import java.io.InputStream;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;

import nil.ucm.indications2.core.rep.IndicationStructureRep;
import nil.ucm.indications2.core.rep.indication.IndicationRep;
import nil.ucm.indications2.core.rep.indication.IndicationTranslationFactory;
import nil.ucm.indications2.ui.dialogs.SelectResourcesDialog;

import org.eclipse.core.commands.AbstractHandler;
import org.eclipse.core.commands.ExecutionEvent;
import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.commands.IHandler;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.actions.WorkspaceModifyOperation;
import org.eclipse.ui.handlers.HandlerUtil;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

import edu.mit.discourse.core.rep.coref.CorefRep;
import edu.mit.discourse.core.rep.coref.CorefTranslationFactory;
import edu.mit.discourse.core.rep.referent.IReferent;
import edu.mit.discourse.core.rep.referent.ReferentRep;
import edu.mit.discourse.core.rep.refexp.RefExpRep;
import edu.mit.discourse.core.rep.refexp.RefExpTranslationFactory;
import edu.mit.parsing.core.rep.pos.POSRep;
import edu.mit.parsing.core.rep.sentence.SentenceRep;
import edu.mit.story.core.StoryPlugin;
import edu.mit.story.core.desc.IDesc;
import edu.mit.story.core.desc.factory.IDescFactory;
import edu.mit.story.core.measure.IResult;
import edu.mit.story.core.measure.StoryFileExtractor;
import edu.mit.story.core.meta.IMetaDesc;
import edu.mit.story.core.meta.IMetaRep;
import edu.mit.story.core.model.IMutableStoryModel;
import edu.mit.story.core.model.IStoryModel;
import edu.mit.story.core.model.StoryModelExporter;
import edu.mit.story.core.model.StoryModelImporter;
import edu.mit.story.core.model.change.AddRepsWithConfigChange;
import edu.mit.story.core.model.change.IModelChange;
import edu.mit.story.core.model.change.RemoveRepresentationsChange;
import edu.mit.story.core.property.FreeConfiguration;
import edu.mit.story.core.property.IConfiguration;
import edu.mit.story.core.property.IRepConfiguration;
import edu.mit.story.core.property.RepConfiguration;
import edu.mit.story.core.rep.IRep;
import edu.mit.story.core.util.Debug;
import edu.mit.story.core.util.XMLUtils;

/** 
 * TODO: Write comment
 *
 * @author M.A. Finlayson
 * @version $Rev$, $LastChangedDate$
 * @since nil.ucm.indications2.ui 1.0.0
 */
public class ACLExportHandler extends AbstractHandler implements IHandler {

	/* 
	 * (non-Javadoc) @see org.eclipse.core.commands.IHandler#execute(org.eclipse.core.commands.ExecutionEvent)
	 */
	public Object execute(ExecutionEvent event) throws ExecutionException {
		
		// extract active shell
		Shell shell = HandlerUtil.getActiveShell(event);
		if(shell == null) return null;
		
		// prompt user for projects
		SelectResourcesDialog dialog = new SelectResourcesDialog(shell, IResource.PROJECT);
		dialog.setBlockOnOpen(true);
		dialog.setTitle("Export for ACL");
		int result = dialog.open();
		if(result == Dialog.CANCEL) 
			return null;
		
		IProject source = (IProject)dialog.getResourceOne();
		IProject target = (IProject)dialog.getResourceTwo();
		
		
		ExportForACLOperation op = new ExportForACLOperation(source, target);
		
		IWorkbenchWindow window = HandlerUtil.getActiveWorkbenchWindow(event);
		try {
			window.run(true, true, op);
		} catch (InvocationTargetException e) {
			e.printStackTrace();
		} catch (InterruptedException e) {
			e.printStackTrace();
		}
		return null;
	}
	
	protected class ExportForACLOperation extends WorkspaceModifyOperation {

		private final IProject source;
		private final IProject target;
		
		private final ACLExporter exporter = new ACLExporter();

		/**
		 * TODO: Write comment
		 *
		 * @param source
		 * @param target
		 * @since nil.ucm.indications2.ui 1.0.0
		 */
		public ExportForACLOperation(IProject source, IProject target) {
			if(source == null)
				throw new NullPointerException();
			if(target == null)
				throw new NullPointerException();
			if(source == target)
				throw new IllegalArgumentException();
			this.source = source;
			this.target = target;
		}

		/* 
		 * (non-Javadoc) @see org.eclipse.ui.actions.WorkspaceModifyOperation#execute(org.eclipse.core.runtime.IProgressMonitor)
		 */
		@Override
		protected void execute(IProgressMonitor monitor) throws CoreException,
				InvocationTargetException, InterruptedException {
			
			// get files
			List<IResult> errors = new LinkedList<IResult>();
			List<IFile> files = StoryFileExtractor.getInstance().getTargets(source, errors);
			
			// post errors
			for(IResult r : errors)
				Debug.err(r.toString(true));
			
			for(IFile file : files)
				process(file);
		}
		
		/**
		 * TODO: Write comment
		 *
		 * @param file
		 * @since nil.ucm.indications2.ui 1.0.0
		 */
		protected void process(IFile file){
			
			Debug.out("Processing " + file.getName());
			
			
			// create and init model
			List<IResult> results = new ArrayList<IResult>(1);
			IMutableStoryModel model = StoryModelImporter.extractModel(file, results);
			if(model == null){
				StoryPlugin.getDefault().getLog().log(results.get(0));
				return;
			}
			IDescFactory factory;;
			for(IRep rep : model.getSupported()){
				factory = model.getFactory(rep);
				if(factory != null)
					factory.init(model);
			}
			
			// do counts
			int refexps1 = countReferences(model);
			int corefs1 = model.getData().getDescriptions(ReferentRep.getInstance()).size();
			int inds1 = model.getData().getDescriptions(IndicationStructureRep.getInstance()).size();
			
			List<IRep> reps = Arrays.asList(RefExpRep.getInstance(), CorefRep.getInstance(), IndicationRep.getInstance());
			IRepConfiguration factConfig = new RepConfiguration(new FreeConfiguration("noid"));
			factConfig.addAllSupport(reps);
			factConfig.put(RefExpRep.getInstance().getID(), RefExpTranslationFactory.ID);
			factConfig.put(CorefRep.getInstance().getID(), CorefTranslationFactory.ID);
			factConfig.put(IndicationRep.getInstance().getID(), IndicationTranslationFactory.ID);
			
			// add new reps
			IModelChange change = new AddRepsWithConfigChange(reps, null, factConfig);
			model.applyChange(this, change, true);
			
			// remove old reps
			change = new RemoveRepresentationsChange(Arrays.asList(IndicationStructureRep.getInstance(), ReferentRep.getInstance(), POSRep.getInstance(), SentenceRep.getInstance()));
			model.applyChange(this, change, true);
			
			int refexps2 = model.getData().getDescriptions(RefExpRep.getInstance()).size();
			int corefs2 = model.getData().getDescriptions(CorefRep.getInstance()).size();
			int inds2 = model.getData().getDescriptions(IndicationRep.getInstance()).size();
			
			if(refexps1 != refexps2)
				Debug.err("Initial model had " + refexps1 + " referring expressions, but final model had " + refexps2);
			if(corefs1 != corefs2)
				Debug.err("Initial model had " + corefs1 + " co-reference relations, but final model had " + corefs2);
			if(inds1 != inds2)
				Debug.err("Initial model had " + inds1 + " indications, but final model had " + inds2);
			
			
			// new file
			IPath path = file.getProjectRelativePath();
			IFile newFile = target.getFile(path);
			if(newFile.exists())
				throw new IllegalArgumentException();
			
			// write model out
			Document xml = exporter.exportToXML(model);
			InputStream content = XMLUtils.toStream(xml);
			try {
				newFile.create(content, true, null);
			} catch (CoreException e) {
				e.printStackTrace();
			}
			
		}
		
		private int countReferences(IStoryModel model){
			int count = 0;
			IReferent r;
			for(IDesc d : model.getData().getDescriptions(ReferentRep.getInstance())){
				r = (IReferent)d.getData();
				count += r.getReferences().size();
			}
			return count;
		}
		
	}
	
	protected class ACLExporter extends StoryModelExporter {

		/* 
		 * (non-Javadoc) @see edu.mit.story.core.model.StoryModelExporter#exportConfig(org.w3c.dom.Node, edu.mit.story.core.property.IConfiguration, org.w3c.dom.Document)
		 */
		@Override
		public void exportConfig(Node parent, IConfiguration config,
				Document doc) {
			return;
		}

		/* 
		 * (non-Javadoc) @see edu.mit.story.core.model.StoryModelExporter#exportFactory(org.w3c.dom.Node, edu.mit.story.core.desc.factory.IDescFactory, org.w3c.dom.Document)
		 */
		@Override
		public Element exportFactory(Node parent, IDescFactory factory,
				Document doc) {
			return null;
		}

		/* 
		 * (non-Javadoc) @see edu.mit.story.core.model.StoryModelExporter#exportMetaDesc(org.w3c.dom.Node, edu.mit.story.core.meta.IMetaDesc, org.w3c.dom.Document)
		 */
		@Override
		public Element exportMetaDesc(Node parent, IMetaDesc<?> desc, Document doc) {
			return null;
		}

		/* 
		 * (non-Javadoc) @see edu.mit.story.core.model.StoryModelExporter#exportMetaRep(org.w3c.dom.Node, edu.mit.story.core.meta.IMetaRep, org.w3c.dom.Document)
		 */
		@Override
		public Element exportMetaRep(Node parent, IMetaRep<?> rep, Document doc) {
			return null;
		}

		/* 
		 * (non-Javadoc) @see edu.mit.story.core.model.StoryModelExporter#exportDesc(org.w3c.dom.Node, edu.mit.story.core.desc.IDesc, org.w3c.dom.Document)
		 */
		@Override
		public Element exportDesc(Node parent, IDesc desc, Document doc) {
			Element node = super.exportDesc(parent, desc, doc);
			node.removeAttribute(ATTR_USER);
			return node;
		}
		
		
	}

}
