/* Filename: Nucleus.java
 * Creator: Raquel Hervas
 * Format: Java 2 v1.6.0
 * Date created: 28/09/2009
 */
package nil.ucm.indications2.core.rep;

import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import edu.mit.parsing.core.rep.token.TokenRep;
import edu.mit.story.core.align.IAlignedStoryModel;
import edu.mit.story.core.desc.Data;
import edu.mit.story.core.desc.IData;
import edu.mit.story.core.desc.IDesc;
import edu.mit.story.core.desc.ISegment;
import edu.mit.story.core.desc.IStructuredData;
import edu.mit.story.core.desc.Segment;
import edu.mit.story.core.model.IStoryData;
import edu.mit.story.core.model.IStoryModel;
import edu.mit.story.core.position.IHasPosition;
import edu.mit.story.core.position.IHasPositionSet;
import edu.mit.story.core.position.ImmutableHasPositionSet;
import edu.mit.story.core.position.PositionUtils;
import edu.mit.story.core.rep.IRep;

/** 
 *   Implementation of the INucleus interface. This class contains the text corresponding
 * to the nucleus and a set of segments
 *
 * @author Raquel Hervas
 * @version 1.0, (Dec. 29, 2009)
 * @since nil.ucm.indications.core 1.0.0
 */
public class Nucleus implements INucleus {

	private final ImmutableHasPositionSet<ISegment> segments;
	private String text;
	private IIndicationStructure parent;
	private IndicationType type;
	
	public Nucleus(IDesc desc, IStoryData data){
		this(Collections.singleton(new Segment(desc, TokenRep.getInstance(), data)));
	}
	
	public Nucleus(Collection<? extends ISegment> segments){
		this(segments, IndicationType.UNKNOWN);
	}
	
	public Nucleus(Collection<? extends ISegment> segments, IndicationType type){
		if(segments.isEmpty()) throw new IllegalArgumentException();
		if(type == null) throw new NullPointerException();
		this.segments = new ImmutableHasPositionSet<ISegment>(segments);
		this.text = Segment.generateText(segments);
		this.type = type;
	}
	
	/* 
	 * (non-Javadoc) @see nil.ucm.indications.core.rep.INucleus#getSegments()
	 */
	
	public IHasPositionSet<ISegment> getSegments() {
		return segments;
	}

	/* 
	 * (non-Javadoc) @see edu.mit.story.core.util.IHasDisplayText#getDisplayText()
	 */
	
	public String getDisplayText() {
		return text;
	}
	
	/* 
	 * (non-Javadoc) @see edu.mit.story.core.desc.IStructuredData#recalculate(edu.mit.story.core.model.IStoryModel)
	 */
	
	public IStructuredData recalculate(IDesc container, IStoryModel model) {
		boolean changed = false;
		Set<ISegment> newSegments = new HashSet<ISegment>(segments.size());
		ISegment newSegment;
		for(ISegment oldSegment : segments){
			newSegment = (ISegment)oldSegment.recalculate(container, model);
			if(newSegment != null) newSegments.add(newSegment);
			changed |= (newSegment != oldSegment);
		}
		return changed ? (newSegments.isEmpty() ? null : new Nucleus(newSegments, type)) : this;
	}

	/* 
	 * (non-Javadoc) @see edu.mit.story.core.desc.IStructuredData#calculatePosition()
	 */
	
	public IHasPosition calculatePosition() {
		return PositionUtils.makePosition(segments);
	}

	/* 
	 * (non-Javadoc) @see edu.mit.story.core.desc.IStructuredData#getDisplayPosition()
	 */
	
	public IHasPosition getDisplayPosition() {
		return calculatePosition();
	}

	/* 
	 * (non-Javadoc) @see nil.ucm.indications.core.rep.INucleus#getParent()
	 */
	
	public IIndicationStructure getParent() {
		if(parent == null) throw new IllegalStateException();
		return parent;
	}
	
	public IndicationType getType() {
		return type;
	}
	
	public void setParent(IIndicationStructure p){
		if(parent != null) throw new IllegalArgumentException();
		parent = p;
	}
	
	public String toString(){
		return "Nuc(" + parent.getReferentDescription().getID() + ':' + parent.getReference().getID() + ")[" + text + "]@" + PositionUtils.toString(this);
	}
	
	/* 
	 * (non-Javadoc) @see edu.mit.story.core.position.IHasPosition#getLength()
	 */
	public int getLength() {
		return segments.getLength();
	}

	/* 
	 * (non-Javadoc) @see edu.mit.story.core.position.IHasPosition#getOffset()
	 */
	public int getOffset() {
		return segments.getOffset();
	}

	/* 
	 * (non-Javadoc) @see edu.mit.story.core.position.IHasPosition#getRightOffset()
	 */
	public int getRightOffset() {
		return segments.getRightOffset();
	}
	
	/* 
	 * (non-Javadoc) @see edu.mit.story.core.rep.IHasRep#getRep()
	 */
	public IRep getRep() {
		return segments.first().getRep();
	}
	
	/* 
	 * (non-Javadoc) @see edu.mit.story.core.desc.IData#equals(edu.mit.story.core.desc.IData, edu.mit.story.core.align.IAlignedStoryModel)
	 */
	public boolean equals(IData tgtData, IAlignedStoryModel model) {
		if(this == tgtData)
			return true;
		if(!INucleus.class.isAssignableFrom(tgtData.getClass()))
			return false;
		INucleus tgtCon = (INucleus)tgtData;
		
		if(!getType().equals(tgtCon.getType()))
			return false;
		if(!Data.equals(segments, tgtCon.getSegments(), model))
			return false;
		
		return true;
	}

}
